﻿[CmdletBinding()]
param(
	[parameter(Mandatory = $false, HelpMessage = "Silent")]
	[Switch]$Force,
	[string]$secure_password
)

# Compressed file name
$CPname = "php-mul-latest-master"

# Signature GPG KEY-ID
$GpgKI = "DBBC8D7BB64C4648A70AEA180FEBF674EAD23E05"

# Save compressed file to
$CPSaveTo = "$($PSScriptRoot)\downloads"

# Temporary directory for compressed files
$TempFolder = "$($env:userprofile)\Downloads\Temp.$($CPname)"

# Generating compressed archive format
# To generate gz or xz files, a tar archive must be created first; otherwise, creation will fail.
$BuildType = @(
	[ZipFormat]::zip
	[ZipFormat]::tar
	[ZipFormat]::gz
	[ZipFormat]::xz
)

# Exclude files or directories from the compressed archive.
$ArchiveExclude = @(
	"-xr-!learn"
	"-xr-!downloads"
)

Enum ZipFormat
{
	zip
	tar
	xz
	gz
}

function Set-Password {
	Write-Host "`n  Please enter your PGP security password!`n" -ForegroundColor Green
	$script:secure_password = read-host "  Certificate password: "
	Write-Host ""
}

function Wait-Exit {
	param(
		[int]$wait
	)
	Write-Host "`n   Note: This script will automatically exit after $($wait) seconds...." -ForegroundColor Red
	Start-Sleep -s $wait
	exit
}

function Get-Zip {
	if (Test-Path "$env:ProgramFiles\7-Zip\7z.exe") {
		$script:Zip = "$env:ProgramFiles\7-Zip\7z.exe"
		return
	}

	if (Test-Path "$env:ProgramFiles(x86)\7-Zip\7z.exe") {
		$script:Zip = "$env:ProgramFiles(x86)\7-Zip\7z.exe"
		return
	}

	$script:Zip = "No"
}

function Clear-old-file {
	$oldfiles = @(
		"php-mul-latest-master.zip"
		"php-mul-latest-master.zip.asc"
		"php-mul-latest-master.zip.sha256"
		"php-mul-latest-master.tar.gz"
		"php-mul-latest-master.tar.gz.asc"
		"php-mul-latest-master.tar.gz.sha256"
		"php-mul-latest-master.tar.xz"
		"php-mul-latest-master.tar.xz.asc"
		"php-mul-latest-master.tar.xz.sha256"
	)

	foreach ($file in $oldfiles) {
		$fullpath = "$($PSScriptRoot)\downloads\$($file)"
		Remove-Item -path $fullpath -Force -ErrorAction SilentlyContinue
	}
}

function Unpack-packer {
	foreach ($list in $BuildType) {
		Unpack-packer-Create -Type $list
	}
}

function Unpack-packer-Create {
	Param(
		[string]$Type
	)

	Check-SD -chkpath $TempFolder

	switch ($Type) {
		"zip" {
			Write-Host "  * Generating $CPname.zip"
			$arguments = "a", "-tzip", "$TempFolder\$CPname.zip", "$ArchiveExclude", "*.*", "-mcu=on", "-r", "-mx9";
		}
		"tar" {
			Write-Host "  * Generating $CPname.tar"
			$arguments = "a", "$TempFolder\$CPname.tar", "$ArchiveExclude", "*.*", "-r";
		}
		"xz" {
			Write-Host "  * Generating $CPname.tar.xz"
			if (Test-Path "$TempFolder\$CPname.tar") {                
				$arguments = "a", "$TempFolder\$CPname.tar.xz", "$TempFolder\$CPname.tar", "-mf=bcj", "-mx9";
			} else {
				Write-Host "    - Skipping generation, not found $CPname.tar`n"
				return
			}
		}
		"gz" {
			Write-Host "  * Generating $CPname.tar.gz"
			if (Test-Path "$TempFolder\$CPname.tar") {           
				$arguments = "a", "-tgzip", "$TempFolder\$CPname.tar.gz", "$TempFolder\$CPname.tar", "-mx9";
			} else {
				Write-Host "  x Skipping generation, not found $CPname.tar`n"
				return
			}
		}
	}
	Start-Process $script:Zip "$arguments" -Wait -WindowStyle Minimized
	Write-Host "    - Done`n" -ForegroundColor Green
}

function Get-SCFile {
	param(
		$opath,
		$shortname,
		$Report
	)

	$fullnewpath = $opath + "." + $Report

	# Add the direct generation command to determine whether it is ASC or SHA256, and add judgment.
	Switch ($Report)
	{
		"asc" {
			Write-Host "  * Generating $CPname.asc"

			Remove-Item -path "${opath}.sig" -Force -ErrorAction SilentlyContinue
			Remove-Item -path "${opath}.asc" -Force -ErrorAction SilentlyContinue
			if (([string]::IsNullOrEmpty($script:secure_password))) {
				$arguments = @(
					"--local-user",
					$GpgKI,
					"--output",
					"""$($opath).asc""",
					"--detach-sign",
					"""$($opath)"""
				)
			} else {
				$arguments = @(
					"--pinentry-mode",
					"loopback",
					"--passphrase",
					$script:secure_password,
					"--local-user",
					$GpgKI,
					"--output",
					"""$($opath).asc""",
					"--detach-sign",
					"""$($opath)"""
				)
			}

			Start-Process gpg -ArgumentList $arguments -Wait -WindowStyle Minimized
			if (Test-Path "$($opath).asc" -PathType Leaf) {
				Write-Host "    - Done`n" -ForegroundColor Green
			} else {
				write-host "    - Failed`n" -ForegroundColor Red
			}
		}
		"sha256" {
			Write-Host "  * Generating $CPname.sha256"
			$calchash = (Get-FileHash $opath -Algorithm SHA256)
			Remove-Item -path $fullnewpath -Force -ErrorAction SilentlyContinue
			$calchash.hash + "  " + $shortname | Out-File -FilePath $fullnewpath -Encoding ASCII
			Write-Host "    - Done`n" -ForegroundColor Green
		}
	}
}

function Create-allfileAS {
	remove-item -path "$TempFolder\*.tar" -force -ErrorAction SilentlyContinue

	Get-ChildItem $TempFolder -Recurse -ErrorAction SilentlyContinue | Foreach-Object {
		Get-SCFile -opath $_.FullName -shortname $_.Name -Report "asc"
		Get-SCFile -opath $_.FullName -shortname $_.Name -Report "sha256"
	}
}

function Check-SD {
	Param(
		[string]$chkpath
	)

	if(!(Test-Path $chkpath -PathType Container)) {
		New-Item -Path $chkpath -ItemType Directory -ErrorAction SilentlyContinue | Out-Null
		if(!(Test-Path $chkpath -PathType Container)) {
			Write-Host "    - Creage folder Failed：$($chkpath)`n" -ForegroundColor Red
			return
		}
	}
}

function Move-allfile {
	Check-SD -chkpath $CPSaveTo

	Get-ChildItem $TempFolder -Recurse -ErrorAction SilentlyContinue | Foreach-Object {
		Move-Item -Path $_.FullName -Destination $CPSaveTo -ErrorAction SilentlyContinue | Out-Null
	}
	remove-item -path "$TempFolder" -Recurse -force -ErrorAction SilentlyContinue
}

function Get-Mainpage {
	clear
	Write-host "
  Author: Yi ( https://fengyi.tel )

  From: Yi's Solution
  buildstring: 26.bk_release.260126-1208

  Task Schedule: 

     1. Delete old files
     2. Package all files ( excluding excluded items )
     3. Generate $CPname ( .zip, tar.gz, tar.xz )
     4. Generate ASC signature file
     5. Generate sha256`n"

	If ($Force) {
		Clear-old-file
		Unpack-packer
		Create-allfileAS
		Move-allfile
	} else {
		Set-Password
		Clear-old-file
		Unpack-packer
		Create-allfileAS
		Move-allfile
		Wait-Exit -wait 6
	}
}

Get-Zip
Get-Mainpage